/**
 * UpSolution Element: List Filter
 */
! function( $, _undefined ) {
	"use strict";

	let lastResult = {};

	/**
	 * @param {Node} container.
	 */
	function usListFilter( container ) {
		let self = this;

		// Private "Variables"
		self.settings = {
			mobileWidth: 600,
		};
		self.result = {};

		/**
		 * @var {{}} Bondable events.
		 */
		self._events = {
			filterСhanged: self.filterСhanged.bind( self ),
			resetFilter: self.resetFilter.bind( self ),
			resultСhanged: $ush.debounce( self.resultСhanged.bind( self ), 1 ),
			openSection: self.openSection.bind( self ),
			openMobileFilters: self.openMobileFilters.bind( self ),
			closeMobileFilters: self.closeMobileFilters.bind( self ),
			resize: $ush.debounce( self.resize.bind( self ), 10 ),
		};

		// Elements
		self.$container = $( container );
		self.$pageContent = $( 'main#page-content' );

		// Gets element settings
		if ( self.$container.is( '[onclick]' ) ) {
			$.extend( self.settings, self.$container[0].onclick() || {} );
		}

		// Events
		$( '.w-filter-item', container )
			.on( 'change', 'input, select', self._events.filterСhanged )
			.on( 'click', '.w-filter-item-reset', self._events.resetFilter )
			.on( 'click', '.w-filter-item-title', self._events.openSection );
		self.$container
			.on( 'click', '.w-filter-opener', self._events.openMobileFilters )
			.on( 'click', '.w-filter-list-closer', self._events.closeMobileFilters );
		$us.$window
			.on( 'resize load', self._events.resize )

		self.on( 'resultСhanged', self._events.resultСhanged );
	}

	// List Filter API
	$.extend( usListFilter.prototype, $ush.mixinEvents, {

		/**
		 * Generates a result from filters.
		 *
		 * @event handler
		 * @param {Event} e The Event interface represents an event which takes place in the DOM.
		 */
		filterСhanged: function( e ) {
			let self = this,
				$filter = $( e.delegateTarget ),
				source = $ush.toString( $filter.data( 'source' ) ),
				value = e.target.value;

			if ( $filter.data( 'source-compare' ) ) {
				source = source + '|' + $filter.data( 'source-compare' );
			}

			if ( $filter.hasClass( 'type_checkbox' ) ) {
				let values = $ush.toArray( self.result[ source ] );
				if ( e.target.checked ) {
					values.push( value );
				} else {
					values = values.filter( ( val ) => { return val !== value } );
				}
				if ( ! values.length ) {
					delete self.result[ source ];
				} else {
					self.result[ source ] = values;
				}

				// Type: Dropdown and Radio buttons
			} else {
				if ( value === '*' ) {
					delete self.result[ source ];
				} else {
					self.result[ source ] = value;
				}
			}
			$filter.toggleClass( 'has_value', !! self.result[ source ] );

			self.trigger( 'resultСhanged' );
		},

		/**
		 * Sends result to the first Post/Product List.
		 *
		 * @event handler
		 */
		resultСhanged: function() {
			let self = this;
			if ( self.result === lastResult ) {
				return;
			}
			lastResult = $ush.clone( self.result );

			let $firstList = $( `
				.w-grid.us_post_list:not(.pagination_numbered):visible,
				.w-grid.us_product_list:not(.pagination_numbered):visible,
				.w-grid-none:visible
			`, self.$pageContent ).first();

			$firstList
				.addClass( 'used_by_list_filter' )
				.trigger( 'usListFilter', [ lastResult, 'list_filter' ] );
		},

		/**
		 * Reset filter values.
		 *
		 * @event handler
		 * @param {Event} e The Event interface represents an event which takes place in the DOM.
		 */
		resetFilter: function( e ) {
			e.stopPropagation();

			let self = this,
				$filter = $( e.delegateTarget ),
				source = $ush.toString( $filter.data( 'source' ) );

			if ( $filter.data( 'source-compare' ) ) {
				source = source + '|' + $filter.data( 'source-compare' );
			}

			if ( $filter.hasClass( 'type_checkbox' ) ) {
				$( 'input', $filter ).prop( 'checked', false );

			} else if ( $filter.hasClass( 'type_radio' ) ) {
				$( 'input:first', $filter ).prop( 'checked', true );

			} else if ( $filter.hasClass( 'type_dropdown' ) ) {
				$( 'select', $filter ).prop( 'selectedIndex', 0 );
			}

			$filter.removeClass( 'has_value' );
			delete self.result[ source ];

			self.trigger( 'resultСhanged' );
		},

		/**
		 * Opens a section.
		 *
		 * @event handler
		 * @param {Event} e The Event interface represents an event which takes place in the DOM.
		 */
		openSection: function( e ) {
			if ( this.$container.hasClass( 'togglable' ) ) {
				let $filter = $( e.delegateTarget );
				$filter.toggleClass( 'open', ! $filter.hasClass( 'open' ) );
			}
		},

		/**
		 * Opens mobile version.
		 *
		 * @event handler
		 * @param {Event} e The Event interface represents an event which takes place in the DOM.
		 */
		openMobileFilters: function( e ) {
			$us.$body.addClass( 'us_filter_open' );
			this.$container.addClass( 'open' );
		},

		/**
		 * Close mobile version.
		 *
		 * @event handler
		 * @param {Event} e The Event interface represents an event which takes place in the DOM.
		 */
		closeMobileFilters: function() {
			$us.$body.removeClass( 'us_filter_open' );
			this.$container.removeClass( 'open' );
		},

		/**
		 * Enables or disables the mobile version.
		 *
		 * @event handler
		 */
		resize: function() {
			let self = this,
				isMobile = $ush.parseInt( $us.$window.width() ) <= $ush.parseInt( self.settings.mobileWidth );

			if ( ! self.$container.hasClass( `state_${ isMobile ? 'mobile' : 'desktop' }` ) ) {
				self.$container.usMod( 'state', isMobile ? 'mobile' : 'desktop' );
				if ( ! isMobile ) {
					$us.$body.removeClass( 'us_filter_open' );
					self.$container.removeClass( 'open' );
				}
			}
		},

	} );

	$.fn.usListFilter = function() {
		return this.each( function() {
			$( this ).data( 'usListFilter', new usListFilter( this ) );
		} );
	};

	$( function() {
		$( '.w-filter.for_list' ).usListFilter();
	} );

}( jQuery );
