<?php defined( 'ABSPATH' ) OR die( 'This script cannot be accessed directly.' );

/**
 * This file applies fallback functions for theme elements and settings
 * NOTE: most functions are applied via 3 filters:
 *
 * `vc_edit_form_fields_attributes_{shortcode}` changes values when editing an element with WPBakery Builder
 * `us_edit_atts_fallback_{shortcode}` changes values when editing an element with Live Builder
 * `us_fallback_atts_{shortcode}` changes values in the element template on the front-end
 */

if ( ! function_exists( 'us_fallback_link_format' ) ) {
	/**
	 * Transforms old Link formats to the new format
	 */
	function us_fallback_link_format( $link ) {
		$link_array = array();

		if ( is_array( $link ) ) {
			$link_array = $link;

			// if it is string and begins with "url:", use the WPBakery way to create array
		} elseif ( strpos( $link, 'url:' ) === 0 OR strpos( $link, '|' ) !== FALSE ) {
			foreach ( explode( '|', $link ) as $link_attr ) {
				$param = explode( ':', $link_attr, 2 );
				if ( ! empty( $param[0] ) AND isset( $param[1] ) ) {
					$link_array[ $param[0] ] = rawurldecode( $param[1] );
				}
			}

			// JSON string + urlencode generated by usof.field
		} elseif ( strpos( rawurldecode( $link ), '{' ) === 0 ) {
			$link = json_decode( rawurldecode( $link ), /* as array */TRUE );
			if ( empty( $link ) ) {
				$link_array['url'] = $link;
			}

			// in other cases use the value as an url
		} else {
			$link_array['url'] = $link;
		}

		// Transform the url {{custom_field}} into "custom field" type
		if ( ! empty( $link_array['url'] ) AND preg_match( "/{{([^}]+)}}/", rawurldecode( $link_array['url'] ), $matches ) ) {
			$link_array['type'] = 'custom_field';

			// We need to replace cases "option|field_name" to "option/field_name" for correct appearance in the link control interface
			$link_array['custom_field'] = str_replace( '|', '/', $matches[1] );
			unset( $link_array['url'] );
		}

		// Force "nofollow" attribute
		if ( ! empty( $link_array['target'] ) AND ! isset( $link_array['rel'] ) ) {
			$link_array['rel'] = 'nofollow';
		}

		// Return string with new format
		return rawurlencode( json_encode( array_map( 'rawurldecode', $link_array ) ) );
	}
}

if ( ! function_exists( 'us_vc_row_columns_fallback_helper' ) ) {
	/**
	 * Fallback for Column formats
	 */
	function us_vc_row_columns_fallback_helper( $shortcode_base, $content ) {
		$result_atts = array();
		$preg_patern = ( $shortcode_base == 'vc_row_inner' )
			? '~\[vc_column_inner(([^\]]+)?)~'
			: '~\[vc_column((?!\_)([^\]]+)?)~'; // Using (?!\_) in regex to prevent matching vc_column_inner shortcode

		$_cols_widths = array();

		if ( preg_match_all( $preg_patern, $content, $_cols_matches ) ) {
			foreach ( $_cols_matches[0] as $key => $_cols_match ) {
				$_col_width = '1/1'; // default value

				if ( ! empty( $_cols_matches[1][ $key ] ) ) {
					$_col_atts = shortcode_parse_atts( $_cols_matches[1][ $key ] );

					$_col_width = isset( $_col_atts['width'] ) ? $_col_atts['width'] : $_col_width;
				}

				$_cols_widths[] = $_col_width;
			}
		}

		// If all columns are equal, use the denominator of the first column width
		// Example: 1/2 + 1/2 becomes 2
		// Example: 1/1 + 1/1 + 1/1 becomes 1
		// Example: 1/3 + 1/3 + 1/3 becomes 3
		// Example: 1/2 + 1/2 + 1/2 + 1/2 becomes 2
		if ( count( array_unique( $_cols_widths ) ) === 1 ) {
			$result_atts['columns'] = substr( $_cols_widths[0], 2 );

			// If TWO columns widths have the same denominator and differ numerators, combine numerators via "-" sign
			// Example: 1/3 + 2/3 becomes 1-2
			// Example: 2/5 + 3/5 becomes 2-3
			// Example: 1/4 + 1/2 becomes custom
		} elseif ( count( $_cols_widths ) === 2 ) {
			if ( substr( $_cols_widths[0], 2 ) === substr( $_cols_widths[1], 2 ) ) {
				$result_atts['columns'] = substr( $_cols_widths[0], 0, 1 ) . '-' . substr( $_cols_widths[1], 0, 1 );
			} else {
				$result_atts['columns'] = 'custom';
			}

			// Predefined value 1/4 + 1/2 + 1/4
		} elseif ( $_cols_widths === array( '1/4', '1/2', '1/4' ) ) {
			$result_atts['columns'] = '1-2-1';

			// Predefined value 1/5 + 3/5 + 1/5
		} elseif ( $_cols_widths === array( '1/5', '3/5', '1/5' ) ) {
			$result_atts['columns'] = '1-3-1';

			// Predefined value 1/6 + 2/3 + 1/6
		} elseif ( $_cols_widths === array( '1/6', '2/3', '1/6' ) ) {
			$result_atts['columns'] = '1-4-1';

			// If THREE or more columns have differ width, force the "custom" value
			// Example: 1/6 + 2/6 + 3/6 becomes 1fr 2fr 3fr
			// Example: 1/6 + 2/12 + 1/4 + 5/12 becomes 1fr 1fr 1.5fr 2.5fr
		} else {
			$result_atts['columns'] = 'custom';
			$result_atts['columns_layout'] = '';

			foreach ( $_cols_widths as $_width ) {
				$fr = explode( '/', $_width );

				if ( ! empty( $fr[0] ) AND ! empty( $fr[1] ) ) {
					$result_atts['columns_layout'] .= round( (int) $fr[0] / (int) $fr[1] * 6, 1 ) . 'fr ';
				}
			}
		}

		return $result_atts;
	}
}

// Row / Section
if ( ! function_exists( 'us_fallback_atts_vc_row' ) ) {
	add_filter( 'vc_edit_form_fields_attributes_vc_row', 'us_fallback_atts_vc_row', 710, 1 );
	add_filter( 'us_edit_atts_fallback_vc_row', 'us_fallback_atts_vc_row', 710, 2 );
	function us_fallback_atts_vc_row( $atts, $content = '' ) {

		// Shape Divider (after version 7.1)
		if (
			empty( $atts['us_shape_show_top'] )
			AND empty( $atts['us_shape_show_bottom'] )
			AND ! empty( $atts['us_shape'] )
			AND $atts['us_shape'] != 'none'
		) {
			$us_shape_position = ( ! empty( $atts['us_shape_position'] ) )
				? $atts['us_shape_position']
				: 'bottom';
			$atts[ 'us_shape_show_' . $us_shape_position ] = 1;
			$atts[ 'us_shape_' . $us_shape_position ] = $atts['us_shape'];

			if ( ! empty( $atts['us_shape_height'] ) ) {
				$atts[ 'us_shape_height_' . $us_shape_position ] = $atts['us_shape_height'];
			}
			if ( ! empty( $atts['us_shape_color'] ) ) {
				$atts[ 'us_shape_color_' . $us_shape_position ] = $atts['us_shape_color'];
			}
			if ( ! empty( $atts['us_shape_overlap'] ) ) {
				$atts[ 'us_shape_overlap_' . $us_shape_position ] = $atts['us_shape_overlap'];
			}
			if ( ! empty( $atts['us_shape_flip'] ) ) {
				$atts[ 'us_shape_flip_' . $us_shape_position ] = $atts['us_shape_flip'];
			}

			// Removing old shape divider params
			foreach ( array( 'us_shape', 'us_shape_height', 'us_shape_position', 'us_shape_color', 'us_shape_overlap', 'us_shape_flip' ) as $param_name ) {
				$atts[ $param_name ] = '';
			}
		}

		// Row Height (after version 8.0)
		if ( ! empty( $atts['height'] ) AND $atts['height'] == 'full' ) {
			$atts['full_height'] = '1';
			$atts['height'] = 'medium';

			if ( ! empty( $atts['valign'] ) ) {
				$atts['v_align'] = $atts['valign'];
				$atts['valign'] = '';
			} else {
				$atts['v_align'] = 'top';
			}
		}

		// Row Width (after version 8.13)
		if ( ! empty( $atts['width'] ) AND $atts['width'] == '1' ) {
			$atts['width'] = 'full';
		}

		// Columns gap, if NEW columns layout is enabled (after version 8.0)
		if (
			us_get_option( 'live_builder' )
			AND us_get_option( 'grid_columns_layout' )
			AND empty( $atts['columns_gap'] )
		) {
			if ( ! empty( $atts['columns_type'] ) ) {

				if ( ! empty( $atts['gap'] ) AND preg_match( '~^(\d*\.?\d*)(.*)$~', $atts['gap'], $matches ) ) {
					$atts['columns_gap'] = ( $matches[1] * 2 ) . $matches[2];
				} else {
					$atts['columns_gap'] = '0rem';
				}

			} elseif ( ! empty( $atts['gap'] ) ) {

				// Avoid "calc" in the value
				if ( strpos( $atts['gap'], 'rem' ) !== FALSE ) {
					$atts['columns_gap'] = ( 3 + (float) $atts['gap'] ) . 'rem';
				} elseif ( strpos( $atts['gap'], 'px' ) !== FALSE ) {
					$atts['columns_gap'] = ( 48 + (int) $atts['gap'] ) . 'px';
				}
			}

			// Always reset the gap attribute when 'grid_columns_layout' is enabled
			if ( ! empty( $atts['gap'] ) ) {
				$atts['gap'] = '';
			}
		}

		return $atts;
	}
}

// Inner Row editing fallback
if ( ! function_exists( 'us_fallback_atts_vc_row_inner' ) ) {
	add_filter( 'vc_edit_form_fields_attributes_vc_row_inner', 'us_fallback_atts_vc_row_inner', 710, 1 );
	add_filter( 'us_edit_atts_fallback_vc_row_inner', 'us_fallback_atts_vc_row_inner', 710, 2 );
	function us_fallback_atts_vc_row_inner( $atts, $content = '' ) {

		// Columns gap, if NEW columns layout is enabled (after version 8.0)
		if (
			us_get_option( 'live_builder' )
			AND us_get_option( 'grid_columns_layout' )
			AND empty( $atts['columns_gap'] )
		) {
			if ( ! empty( $atts['columns_type'] ) ) {

				if ( ! empty( $atts['gap'] ) AND preg_match( '~^(\d*\.?\d*)(.*)$~', $atts['gap'], $matches ) ) {
					$atts['columns_gap'] = ( $matches[1] * 2 ) . $matches[2];
				} else {
					$atts['columns_gap'] = '0rem';
				}

			} elseif ( ! empty( $atts['gap'] ) ) {

				// Avoid "calc" in the value
				if ( strpos( $atts['gap'], 'rem' ) !== FALSE ) {
					$atts['columns_gap'] = ( 3 + (float) $atts['gap'] ) . 'rem';
				} elseif ( strpos( $atts['gap'], 'px' ) !== FALSE ) {
					$atts['columns_gap'] = ( 48 + (int) $atts['gap'] ) . 'px';
				}
			}

			// Always reset the gap attribute when 'grid_columns_layout' is enabled
			if ( ! empty( $atts['gap'] ) ) {
				$atts['gap'] = '';
			}
		}

		return $atts;
	}
}

// Row/Inner Row output or editing fallback
if ( ! function_exists( 'usb_fallback_vc_row' ) ) {
	add_filter( 'us_fallback_atts_vc_row', 'usb_fallback_vc_row', 800, 2 );
	add_filter( 'us_fallback_atts_vc_row_inner', 'usb_fallback_vc_row', 800, 2 );

	add_filter( 'us_edit_atts_fallback_vc_row', 'usb_fallback_vc_row', 800, 2 );
	add_filter( 'us_edit_atts_fallback_vc_row_inner', 'usb_fallback_vc_row', 800, 2 );

	function usb_fallback_vc_row( $atts, $content = '' ) {
		if ( empty( $atts ) ) {
			return $atts;
		}

		$shortcode_base = (
			strpos( current_filter(), 'vc_row_inner' ) !== FALSE
				? 'vc_row_inner'
				: 'vc_row'
		);

		// For compatibility of columns Edit Live and WPBakery Page Builder
		$columns_fallback_result = us_vc_row_columns_fallback_helper( $shortcode_base, $content );
		if (
			! empty( $columns_fallback_result['columns'] )
			AND ! (
				! empty( $atts['columns'] )
				AND $atts['columns'] == 'custom'
			)
		) {
			$atts['columns'] = $columns_fallback_result['columns'];
		}
		if ( ! empty( $columns_fallback_result['columns_layout'] ) ) {
			$atts['columns_layout'] = $columns_fallback_result['columns_layout'];
		}

		return $atts;
	}
}

// Column & Inner Column
if ( ! function_exists( 'us_fallback_atts_vc_column' ) ) {

	// Column
	add_filter( 'vc_edit_form_fields_attributes_vc_column', 'us_fallback_atts_vc_column', 710 );
	add_filter( 'us_edit_atts_fallback_vc_column', 'us_fallback_atts_vc_column', 710 );
	add_filter( 'us_fallback_atts_vc_column', 'us_fallback_atts_vc_column', 710 );

	// Inner Column
	add_filter( 'vc_edit_form_fields_attributes_vc_column_inner', 'us_fallback_atts_vc_column', 710 );
	add_filter( 'us_edit_atts_fallback_vc_column_inner', 'us_fallback_atts_vc_column', 710 );
	add_filter( 'us_fallback_atts_vc_column_inner', 'us_fallback_atts_vc_column', 710 );

	function us_fallback_atts_vc_column( $atts ) {

		// Link format (after version 8.16)
		if (
			! empty( $atts['link'] )
			AND strpos( rawurldecode( $atts['link'] ), '{"' ) !== 0 // this check ignores new link format
		) {
			$atts['link'] = us_fallback_link_format( $atts['link'] );
		}

		return $atts;
	}
}

// Column & Inner Column design settings
// Fallback "css" attribute separately, because it can work in the element editing only
if ( ! function_exists( 'us_fallback_edit_atts_vc_column' ) ) {

	// Column
	add_filter( 'vc_edit_form_fields_attributes_vc_column', 'us_fallback_edit_atts_vc_column', 710 );
	add_filter( 'us_edit_atts_fallback_vc_column', 'us_fallback_edit_atts_vc_column', 710 );

	// Inner Column
	add_filter( 'vc_edit_form_fields_attributes_vc_column_inner', 'us_fallback_edit_atts_vc_column', 710 );
	add_filter( 'us_edit_atts_fallback_vc_column_inner', 'us_fallback_edit_atts_vc_column', 710 );

	function us_fallback_edit_atts_vc_column( $atts ) {

		// Animation (after version 8.0)
		if ( ! empty( $atts['animate'] ) ) {
			if ( ! empty( $atts['css'] ) ) {
				if ( is_string( $atts['css'] ) ) {
					$css_arr = json_decode( rawurldecode( $atts['css'] ), TRUE );
				} else {
					$css_arr = $atts['css'];
				}
			} else {
				$css_arr = array();
			}

			if ( ! is_array( $css_arr ) ) {
				$css_arr = array();
			}
			if ( empty( $css_arr['default']['animation-name'] ) ) {
				if ( ! isset( $css_arr['default'] ) ) {
					$css_arr['default'] = array();
				}
				$css_arr['default']['animation-name'] = $atts['animate'];

				if ( ! empty( $atts['animate_delay'] ) ) {
					$css_arr['default']['animation-delay'] = (float) $atts['animate_delay'] . 's';
				}

				$atts['css'] = rawurlencode( json_encode( $css_arr ) );
			}

			$atts['animate'] = '';
			$atts['animate_delay'] = '';
		}

		return $atts;
	}
}

// Button
if ( ! function_exists( 'us_fallback_atts_us_btn' ) ) {
	add_filter( 'vc_edit_form_fields_attributes_us_btn', 'us_fallback_atts_us_btn', 710, 1 );
	add_filter( 'us_edit_atts_fallback_us_btn', 'us_fallback_atts_us_btn', 710, 2 );
	add_filter( 'us_fallback_atts_us_btn', 'us_fallback_atts_us_btn', 710, 1 );
	function us_fallback_atts_us_btn( $atts, $content = '' ) {

		// Stretch (after version 8.10)
		if ( ! empty( $atts['width_type'] ) ) {
			$atts['align'] = 'justify';
			$atts['width_type'] = '';
		}

		// Link format (after version 8.16)
		if ( ! empty( $atts['link_type'] ) ) {
			if ( $atts['link_type'] == 'none' ) {
				$link_array = array( 'url' => '' );

			} elseif ( in_array( $atts['link_type'], array( 'post', 'archive', 'elm_value', 'popup_image', 'popup_post' ) ) ) {
				$link_array = array( 'type' => $atts['link_type'] );

			} elseif ( $atts['link_type'] == 'onclick' AND ! empty( $atts['onclick_code'] ) ) {
				$link_array = array( 'url' => '#', 'onclick' => $atts['onclick_code'] );
				$atts['onclick_code'] = '';

			} elseif ( $atts['link_type'] != 'custom' ) {
				$link_array = array( 'type' => 'custom_field', 'custom_field' => $atts['link_type'] );
				$atts['hide_with_empty_link'] = '1';
			}

			if ( ! empty( $atts['link_new_tab'] ) ) {
				$link_array['target'] = '_blank';
				$link_array['rel'] = 'nofollow';
				$atts['link_new_tab'] = '';
			}

			if ( ! empty( $link_array ) ) {
				$atts['link'] = us_fallback_link_format( $link_array );
			}

			$atts['link_type'] = '';
		}

		if (
			! empty( $atts['link'] )
			AND (
				is_array( $atts['link'] )
				OR strpos( rawurldecode( $atts['link'] ), '{"' ) !== 0 // this check ignores new link format
			)
		) {
			$atts['link'] = us_fallback_link_format( $atts['link'] );
			$atts['hide_with_empty_link'] = '1';
		}

		return $atts;
	}
}

// Post Custom Field
if ( ! function_exists( 'us_fallback_atts_us_post_custom_field' ) ) {
	add_filter( 'vc_edit_form_fields_attributes_us_post_custom_field', 'us_fallback_atts_us_post_custom_field', 710, 1 );
	add_filter( 'us_edit_atts_fallback_us_post_custom_field', 'us_fallback_atts_us_post_custom_field', 710, 2 );
	add_filter( 'us_fallback_atts_us_post_custom_field', 'us_fallback_atts_us_post_custom_field', 710, 1 );
	function us_fallback_atts_us_post_custom_field( $atts, $content = '' ) {

		// Link format (after version 8.16)
		if (
			! empty( $atts['link'] )
			AND strpos( rawurldecode( $atts['link'] ), '{"' ) !== 0 // this check ignores new link format
		) {
			if ( $atts['link'] == 'popup_post_image' ) {
				$link_array = array( 'type' => 'popup_image' );

			} elseif ( in_array( $atts['link'], array( 'post', 'elm_value' ) ) ) {
				$link_array = array( 'type' => $atts['link'] );

			} elseif ( $atts['link'] == 'onclick' AND ! empty( $atts['onclick_code'] ) ) {
				$link_array = array( 'url' => '#', 'onclick' => $atts['onclick_code'] );
				$atts['onclick_code'] = '';

			} elseif ( $atts['link'] != 'custom' ) {
				$link_array = array( 'type' => 'custom_field', 'custom_field' => $atts['link'] );
			}

			if ( ! empty( $atts['link_new_tab'] ) ) {
				$link_array['target'] = '_blank';
				$link_array['rel'] = 'nofollow';
				$atts['link_new_tab'] = '';
			}

			if ( ! empty( $link_array ) ) {
				$atts['link'] = us_fallback_link_format( $link_array );

			} elseif ( ! empty( $atts['custom_link'] ) ) {
				$atts['link'] = us_fallback_link_format( $atts['custom_link'] );
				$atts['custom_link'] = '';
			}
		}

		return $atts;
	}
}

// Post Title
if ( ! function_exists( 'us_fallback_atts_us_post_title' ) ) {
	add_filter( 'vc_edit_form_fields_attributes_us_post_title', 'us_fallback_atts_us_post_title', 710, 1 );
	add_filter( 'us_edit_atts_fallback_us_post_title', 'us_fallback_atts_us_post_title', 710, 2 );
	add_filter( 'us_fallback_atts_us_post_title', 'us_fallback_atts_us_post_title', 710, 1 );
	function us_fallback_atts_us_post_title( $atts, $content = '' ) {

		// Link format (after version 8.16)
		if (
			! empty( $atts['link'] )
			AND strpos( rawurldecode( $atts['link'] ), '{"' ) !== 0 // this check ignores new link format
		) {
			if ( $atts['link'] == 'none' ) {
				$link_array = array( 'url' => '' );

			} elseif ( $atts['link'] == 'post' ) {
				$link_array = array( 'type' => 'post' );

			} elseif ( $atts['link'] != 'custom' ) {
				$link_array = array( 'type' => 'custom_field', 'custom_field' => $atts['link'] );
			}

			if ( ! empty( $atts['link_new_tab'] ) ) {
				$link_array['target'] = '_blank';
				$link_array['rel'] = 'nofollow';
				$atts['link_new_tab'] = '';
			}

			if ( ! empty( $link_array ) ) {
				$atts['link'] = us_fallback_link_format( $link_array );

			} elseif ( ! empty( $atts['custom_link'] ) ) {
				$atts['link'] = us_fallback_link_format( $atts['custom_link'] );
				$atts['custom_link'] = '';
			}
		}

		return $atts;
	}
}

// Post Taxonomy
if ( ! function_exists( 'us_fallback_atts_us_post_taxonomy' ) ) {
	add_filter( 'vc_edit_form_fields_attributes_us_post_taxonomy', 'us_fallback_atts_us_post_taxonomy', 710, 1 );
	add_filter( 'us_edit_atts_fallback_us_post_taxonomy', 'us_fallback_atts_us_post_taxonomy', 710, 2 );
	add_filter( 'us_fallback_atts_us_post_taxonomy', 'us_fallback_atts_us_post_taxonomy', 710, 1 );
	function us_fallback_atts_us_post_taxonomy( $atts, $content = '' ) {

		// Link format (after version 8.16)
		if (
			! empty( $atts['link'] )
			AND strpos( rawurldecode( $atts['link'] ), '{"' ) !== 0 // this check ignores new link format
		) {
			if ( $atts['link'] == 'none' ) {
				$link_array = array( 'url' => '' );

			} elseif ( in_array( $atts['link'], array( 'post', 'archive' ) ) ) {
				$link_array = array( 'type' => $atts['link'] );
			}

			if ( ! empty( $link_array ) ) {
				$atts['link'] = us_fallback_link_format( $link_array );

			} elseif ( ! empty( $atts['custom_link'] ) ) {
				$atts['link'] = us_fallback_link_format( $atts['custom_link'] );
				$atts['custom_link'] = '';
			}
		}

		return $atts;
	}
}

// Post Image
if ( ! function_exists( 'us_fallback_atts_us_post_image' ) ) {
	add_filter( 'vc_edit_form_fields_attributes_us_post_image', 'us_fallback_atts_us_post_image', 710, 1 );
	add_filter( 'us_edit_atts_fallback_us_post_image', 'us_fallback_atts_us_post_image', 710, 2 );
	add_filter( 'us_fallback_atts_us_post_image', 'us_fallback_atts_us_post_image', 710, 1 );
	function us_fallback_atts_us_post_image( $atts, $content = '' ) {

		// Link format (after version 8.16)
		if (
			! empty( $atts['link'] )
			AND strpos( rawurldecode( $atts['link'] ), '{"' ) !== 0 // this check ignores new link format
		) {
			if ( $atts['link'] == 'popup_post_image' ) {
				$link_array = array( 'type' => 'popup_image' );

			} elseif ( $atts['link'] == 'post' ) {
				$link_array = array( 'type' => 'post' );

			} elseif ( $atts['link'] != 'custom' ) {
				$link_array = array( 'type' => 'custom_field', 'custom_field' => $atts['link'] );
			}

			if ( ! empty( $atts['link_new_tab'] ) ) {
				$link_array['target'] = '_blank';
				$link_array['rel'] = 'nofollow';
				$atts['link_new_tab'] = '';
			}

			if ( ! empty( $link_array ) ) {
				$atts['link'] = us_fallback_link_format( $link_array );

			} elseif ( ! empty( $atts['custom_link'] ) ) {
				$atts['link'] = us_fallback_link_format( $atts['custom_link'] );
				$atts['custom_link'] = '';
			}
		}

		return $atts;
	}
}

// Post Author
if ( ! function_exists( 'us_fallback_atts_us_post_author' ) ) {
	add_filter( 'vc_edit_form_fields_attributes_us_post_author', 'us_fallback_atts_us_post_author', 710, 1 );
	add_filter( 'us_edit_atts_fallback_us_post_author', 'us_fallback_atts_us_post_author', 710, 2 );
	add_filter( 'us_fallback_atts_us_post_author', 'us_fallback_atts_us_post_author', 710, 1 );
	function us_fallback_atts_us_post_author( $atts, $content = '' ) {

		// Link format (after version 8.16)
		if (
			! empty( $atts['link'] )
			AND strpos( rawurldecode( $atts['link'] ), '{"' ) !== 0 // this check ignores new link format
		) {
			if ( $atts['link'] == 'none' ) {
				$link_array = array( 'url' => '' );

			} elseif ( in_array( $atts['link'], array( 'post', 'author_page', 'author_website' ) ) ) {
				$link_array = array( 'type' => $atts['link'] );

			} elseif ( $atts['link'] != 'custom' ) {
				$link_array = array( 'type' => 'custom_field', 'custom_field' => $atts['link'] );
			}

			if ( ! empty( $link_array ) ) {
				$atts['link'] = us_fallback_link_format( $link_array );

			} elseif ( ! empty( $atts['custom_link'] ) ) {
				$atts['link'] = us_fallback_link_format( $atts['custom_link'] );
				$atts['custom_link'] = '';
			}
		}

		return $atts;
	}
}

// Post Comments
if ( ! function_exists( 'us_fallback_atts_us_post_comments' ) ) {
	add_filter( 'vc_edit_form_fields_attributes_us_post_comments', 'us_fallback_atts_us_post_comments', 710, 1 );
	add_filter( 'us_edit_atts_fallback_us_post_comments', 'us_fallback_atts_us_post_comments', 710, 2 );
	add_filter( 'us_fallback_atts_us_post_comments', 'us_fallback_atts_us_post_comments', 710, 1 );
	function us_fallback_atts_us_post_comments( $atts, $content = '' ) {

		// Link format (after version 8.16)
		if (
			! empty( $atts['link'] )
			AND strpos( rawurldecode( $atts['link'] ), '{"' ) !== 0 // this check ignores new link format
		) {
			if ( $atts['link'] == 'none' ) {
				$link_array = array( 'url' => '' );

			} elseif ( $atts['link'] == 'post' ) {
				$link_array = array( 'type' => 'post_comments' );

			} elseif ( $atts['link'] != 'custom' ) {
				$link_array = array( 'type' => 'custom_field', 'custom_field' => $atts['link'] );
			}

			if ( ! empty( $link_array ) ) {
				$atts['link'] = us_fallback_link_format( $link_array );

			} elseif ( ! empty( $atts['custom_link'] ) ) {
				$atts['link'] = us_fallback_link_format( $atts['custom_link'] );
				$atts['custom_link'] = '';
			}
		}

		return $atts;
	}
}

// ActionBox (CTA)
if ( ! function_exists( 'us_fallback_atts_us_cta' ) ) {
	add_filter( 'vc_edit_form_fields_attributes_us_cta', 'us_fallback_atts_us_cta', 710, 1 );
	add_filter( 'us_edit_atts_fallback_us_cta', 'us_fallback_atts_us_cta', 710, 2 );
	add_filter( 'us_fallback_atts_us_cta', 'us_fallback_atts_us_cta', 710, 2 );
	function us_fallback_atts_us_cta( $atts, $content = '' ) {

		// Button 1 Link format (after version 8.16)
		if (
			! empty( $atts['btn_link'] )
			AND strpos( rawurldecode( $atts['btn_link'] ), '{"' ) !== 0 // this check ignores new link format
		) {
			$atts['btn_link'] = us_fallback_link_format( $atts['btn_link'] );
		}

		// Button 2 Link format (after version 8.16)
		if (
			! empty( $atts['btn2_link'] )
			AND strpos( rawurldecode( $atts['btn2_link'] ), '{"' ) !== 0 // this check ignores new link format
		) {
			$atts['btn2_link'] = us_fallback_link_format( $atts['btn2_link'] );
		}

		return $atts;
	}
}

// FlipBox
if ( ! function_exists( 'us_fallback_atts_us_flipbox' ) ) {
	add_filter( 'vc_edit_form_fields_attributes_us_flipbox', 'us_fallback_atts_us_flipbox', 710, 1 );
	add_filter( 'us_edit_atts_fallback_us_flipbox', 'us_fallback_atts_us_flipbox', 710, 2 );
	add_filter( 'us_fallback_atts_us_flipbox', 'us_fallback_atts_us_flipbox', 710, 2 );
	function us_fallback_atts_us_flipbox( $atts, $content = '' ) {

		// Link format (after version 8.16)
		if (
			! empty( $atts['link'] )
			AND strpos( rawurldecode( $atts['link'] ), '{"' ) !== 0 // this check ignores new link format
		) {
			$atts['link'] = us_fallback_link_format( $atts['link'] );
		}

		return $atts;
	}
}

// Contact Form
if ( ! function_exists( 'us_fallback_atts_us_cform' ) ) {
	add_filter( 'vc_edit_form_fields_attributes_us_cform', 'us_fallback_atts_us_cform', 710, 1 );
	add_filter( 'us_edit_atts_fallback_us_cform', 'us_fallback_atts_us_cform', 710, 2 );
	add_filter( 'us_fallback_atts_us_cform', 'us_fallback_atts_us_cform', 710, 2 );
	function us_fallback_atts_us_cform( $atts, $content = '' ) {

		// Stretch (after version 8.10)
		if ( ! empty( $atts['button_fullwidth'] ) ) {
			$atts['button_align'] = 'justify';
			$atts['button_fullwidth'] = '';
		}

		return $atts;
	}
}

// Tabs
if ( ! function_exists( 'us_fallback_atts_vc_tta_tabs' ) ) {
	add_filter( 'vc_edit_form_fields_attributes_vc_tta_tabs', 'us_fallback_atts_vc_tta_tabs', 710, 1 );
	add_filter( 'us_edit_atts_fallback_vc_tta_tabs', 'us_fallback_atts_vc_tta_tabs', 710, 2 );
	add_filter( 'us_fallback_atts_vc_tta_tabs', 'us_fallback_atts_vc_tta_tabs', 710, 2 );
	function us_fallback_atts_vc_tta_tabs( $atts, $content = '' ) {

		// Stretch (after version 8.4)
		if ( ! empty( $atts['stretch'] ) ) {
			$atts['tabs_alignment'] = 'justify';
			$atts['stretch'] = '';
		}

		return $atts;
	}
}

// TTA Section
if ( ! function_exists( 'us_fallback_atts_vc_tta_section' ) ) {
	add_filter( 'vc_edit_form_fields_attributes_vc_tta_section', 'us_fallback_atts_vc_tta_section', 710, 1 );
	add_filter( 'us_edit_atts_fallback_vc_tta_section', 'us_fallback_atts_vc_tta_section', 710, 2 );
	add_filter( 'us_fallback_atts_vc_tta_section', 'us_fallback_atts_vc_tta_section', 710, 2 );
	function us_fallback_atts_vc_tta_section( $atts, $content = '' ) {

		// Tab ID (after version 8.0)
		if ( ! empty( $atts['tab_id'] ) ) {
			$atts['el_id'] = $atts['tab_id'];
			$atts['tab_id'] = '';
		}

		// Link format (after version 8.16)
		if (
			! empty( $atts['tab_link'] )
			AND strpos( rawurldecode( $atts['tab_link'] ), '{"' ) !== 0 // this check ignores new link format
		) {
			$atts['tab_link'] = us_fallback_link_format( $atts['tab_link'] );
		}

		return $atts;
	}
}

// Image
if ( ! function_exists( 'us_fallback_atts_us_image' ) ) {
	add_filter( 'vc_edit_form_fields_attributes_us_image', 'us_fallback_atts_us_image', 710 );
	add_filter( 'us_edit_atts_fallback_us_image', 'us_fallback_atts_us_image', 710 );
	add_filter( 'us_fallback_atts_us_image', 'us_fallback_atts_us_image', 710 );
	function us_fallback_atts_us_image( $atts ) {

		// Link format (after version 8.16)
		if ( ! empty( $atts['onclick'] ) ) {
			if ( $atts['onclick'] == 'none' ) {
				$link_array = array( 'url' => '' );

			} elseif ( $atts['onclick'] == 'lightbox' ) {
				$link_array = array( 'type' => 'popup_image' );

			} elseif ( $atts['onclick'] == 'post' ) {
				$link_array = array( 'type' => 'post' );

			} elseif ( $atts['onclick'] == 'onclick' AND ! empty( $atts['onclick_code'] ) ) {
				$link_array = array( 'url' => '#', 'onclick' => $atts['onclick_code'] );
				$atts['onclick_code'] = '';

			} elseif ( $atts['onclick'] != 'custom_link' ) {
				$link_array = array( 'type' => 'custom_field', 'custom_field' => $atts['onclick'] );
			}

			if ( ! empty( $atts['link_new_tab'] ) ) {
				$link_array['target'] = '_blank';
				$link_array['rel'] = 'nofollow';
				$atts['link_new_tab'] = '';
			}

			if ( ! empty( $link_array ) ) {
				$atts['link'] = us_fallback_link_format( $link_array );
			} elseif ( ! empty( $atts['link'] ) ) {
				$atts['link'] = us_fallback_link_format( $atts['link'] );
			}

			$atts['onclick'] = '';
		}

		return $atts;
	}
}

// Image design settings
// Fallback "css" attribute separately, because it can work in the element editing only
if ( ! function_exists( 'us_fallback_edit_atts_us_image' ) ) {
	add_filter( 'vc_edit_form_fields_attributes_us_image', 'us_fallback_edit_atts_us_image', 710 );
	add_filter( 'us_edit_atts_fallback_us_image', 'us_fallback_edit_atts_us_image', 710 );
	function us_fallback_edit_atts_us_image( $atts ) {

		// Animation (after version 8.0)
		if ( ! empty( $atts['animate'] ) ) {
			if ( ! empty( $atts['css'] ) ) {
				if ( is_string( $atts['css'] ) ) {
					$css_arr = json_decode( rawurldecode( $atts['css'] ), TRUE );
				} else {
					$css_arr = $atts['css'];
				}
			} else {
				$css_arr = array();
			}

			if ( ! is_array( $css_arr ) ) {
				$css_arr = array();
			}
			if ( empty( $css_arr['default']['animation-name'] ) ) {
				if ( ! isset( $css_arr['default'] ) ) {
					$css_arr['default'] = array();
				}
				$css_arr['default']['animation-name'] = $atts['animate'];

				if ( ! empty( $atts['animate_delay'] ) ) {
					$css_arr['default']['animation-delay'] = (float) $atts['animate_delay'] . 's';
				}

				$atts['css'] = rawurlencode( json_encode( $css_arr ) );
			}

			$atts['animate'] = '';
			$atts['animate_delay'] = '';
		}

		return $atts;
	}
}

// Text
if ( ! function_exists( 'us_fallback_atts_us_text' ) ) {
	add_filter( 'vc_edit_form_fields_attributes_us_text', 'us_fallback_atts_us_text', 710 );
	add_filter( 'us_edit_atts_fallback_us_text', 'us_fallback_atts_us_text', 710 );
	add_filter( 'us_fallback_atts_us_text', 'us_fallback_atts_us_text', 710 );
	function us_fallback_atts_us_text( $atts ) {

		// Link format (after version 8.16)
		if ( ! empty( $atts['link_type'] ) ) {
			if ( $atts['link_type'] == 'none' ) {
				$link_array = array( 'url' => '' );

			} elseif ( in_array( $atts['link_type'], array( 'post', 'elm_value' ) ) ) {
				$link_array = array( 'type' => $atts['link_type'] );

			} elseif ( $atts['link_type'] == 'onclick' AND ! empty( $atts['onclick_code'] ) ) {
				$link_array = array( 'url' => '#', 'onclick' => $atts['onclick_code'] );
				$atts['onclick_code'] = '';

			} elseif ( $atts['link_type'] != 'custom' ) {
				$link_array = array( 'type' => 'custom_field', 'custom_field' => $atts['link_type'] );
			}

			if ( ! empty( $atts['link_new_tab'] ) ) {
				$link_array['target'] = '_blank';
				$link_array['rel'] = 'nofollow';
				$atts['link_new_tab'] = '';
			}

			if ( ! empty( $link_array ) ) {
				$atts['link'] = us_fallback_link_format( $link_array );
			} elseif ( ! empty( $atts['link'] ) ) {
				$atts['link'] = us_fallback_link_format( $atts['link'] );
			}

			$atts['link_type'] = '';
		}

		return $atts;
	}
}

// Text design settings
// Fallback "css" attribute separately, because it can work in the element editing only
if ( ! function_exists( 'us_fallback_edit_atts_us_text' ) ) {
	add_filter( 'vc_edit_form_fields_attributes_us_text', 'us_fallback_edit_atts_us_text', 710 );
	add_filter( 'us_edit_atts_fallback_us_text', 'us_fallback_edit_atts_us_text', 710 );
	function us_fallback_edit_atts_us_text( $atts ) {

		// Alignment (after version 7.1)
		if ( ! empty( $atts['align'] ) AND $atts['align'] != 'none' ) {
			if ( ! empty( $atts['css'] ) ) {
				if ( is_string( $atts['css'] ) ) {
					$css_arr = json_decode( rawurldecode( $atts['css'] ), TRUE );
				} else {
					$css_arr = $atts['css'];
				}
			} else {
				$css_arr = array();
			}

			if ( ! is_array( $css_arr ) ) {
				$css_arr = array();
			}
			if ( empty( $css_arr['default']['text-align'] ) ) {
				if ( ! isset( $css_arr['default'] ) ) {
					$css_arr['default'] = array();
				}
				$css_arr['default']['text-align'] = $atts['align'];
				$atts['css'] = rawurlencode( json_encode( $css_arr ) );
			}
			$atts['align'] = '';
		}

		return $atts;
	}
}

// Pricing Table
if ( ! function_exists( 'us_fallback_atts_us_pricing' ) ) {
	add_filter( 'vc_edit_form_fields_attributes_us_pricing', 'us_fallback_atts_us_pricing', 810, 1 );
	add_filter( 'us_edit_atts_fallback_us_pricing', 'us_fallback_atts_us_pricing', 810, 2 );
	add_filter( 'us_fallback_atts_us_pricing', 'us_fallback_atts_us_pricing', 810, 2 );
	function us_fallback_atts_us_pricing( $atts, $content = '' ) {
		if ( ! empty( $atts['items'] ) ) {
			$items = is_array( $atts['items'] ) ? $atts['items'] : json_decode( rawurldecode( $atts['items'] ), TRUE );
			foreach ( $items as &$item ) {

				// Featured type
				if ( isset( $item['type'] ) AND $item['type'] === 'featured' ) {
					$item['type'] = '1';
				}

				// Link format (after version 8.16)
				if (
					! empty( $item['btn_link'] )
					AND strpos( rawurldecode( $item['btn_link'] ), '{"' ) !== 0 // this check ignores new link format
				) {
					$item['btn_link'] = us_fallback_link_format( $item['btn_link'] );
				}
			}
			$atts['items'] = rawurlencode( json_encode( $items ) );
			unset( $items );
		}

		return $atts;
	}
}

// Dropdown
if ( ! function_exists( 'us_fallback_atts_us_dropdown' ) ) {
	add_filter( 'vc_edit_form_fields_attributes_us_dropdown', 'us_fallback_atts_us_dropdown', 810, 1 );
	add_filter( 'us_edit_atts_fallback_us_dropdown', 'us_fallback_atts_us_dropdown', 810, 2 );
	add_filter( 'us_fallback_atts_us_dropdown', 'us_fallback_atts_us_dropdown', 810, 2 );
	function us_fallback_atts_us_dropdown( $atts, $content = '' ) {

		if ( ! empty( $atts['links'] ) ) {
			$_group_value_type = is_array( $atts['links'] ) ? 'array' : 'string';
			$items = is_array( $atts['links'] ) ? $atts['links'] : json_decode( rawurldecode( $atts['links'] ), TRUE );
			foreach ( $items as &$item ) {

				// Link format (after version 8.16)
				if (
					! empty( $item['url'] )
					AND ! (
						// check if we already have the new link format
						is_string( $item['url'] )
						AND strpos( rawurldecode( $item['url'] ), '{"' ) === 0
					)
				) {
					$item['url'] = us_fallback_link_format( $item['url'] );
				}
			}
			$atts['links'] = ( $_group_value_type == 'array' ) ? $items : rawurlencode( json_encode( $items ) );
			unset( $items );
		}

		return $atts;
	}
}

// IconBox
if ( ! function_exists( 'us_fallback_atts_us_iconbox' ) ) {
	add_filter( 'vc_edit_form_fields_attributes_us_iconbox', 'us_fallback_atts_us_iconbox', 710, 1 );
	add_filter( 'us_edit_atts_fallback_us_iconbox', 'us_fallback_atts_us_iconbox', 710, 2 );
	add_filter( 'us_fallback_atts_us_iconbox', 'us_fallback_atts_us_iconbox', 710, 2 );
	function us_fallback_atts_us_iconbox( $atts, $content = '' ) {

		// Link format (after version 8.16)
		if (
			! empty( $atts['link'] )
			AND strpos( rawurldecode( $atts['link'] ), '{"' ) !== 0 // this check ignores new link format
		) {
			$atts['link'] = us_fallback_link_format( $atts['link'] );
		}

		return $atts;
	}
}

// Separator
if ( ! function_exists( 'us_fallback_atts_us_separator' ) ) {
	add_filter( 'vc_edit_form_fields_attributes_us_separator', 'us_fallback_atts_us_separator', 710, 1 );
	add_filter( 'us_edit_atts_fallback_us_separator', 'us_fallback_atts_us_separator', 710, 2 );
	add_filter( 'us_fallback_atts_us_separator', 'us_fallback_atts_us_separator', 710, 2 );
	function us_fallback_atts_us_separator( $atts, $content = '' ) {

		// Link format (after version 8.16)
		if (
			! empty( $atts['link'] )
			AND strpos( rawurldecode( $atts['link'] ), '{"' ) !== 0 // this check ignores new link format
		) {
			$atts['link'] = us_fallback_link_format( $atts['link'] );
		}

		return $atts;
	}
}

// Person
if ( ! function_exists( 'us_fallback_atts_us_person' ) ) {
	add_filter( 'vc_edit_form_fields_attributes_us_person', 'us_fallback_atts_us_person', 710, 1 );
	add_filter( 'us_edit_atts_fallback_us_person', 'us_fallback_atts_us_person', 710, 2 );
	add_filter( 'us_fallback_atts_us_person', 'us_fallback_atts_us_person', 710, 2 );
	function us_fallback_atts_us_person( $atts, $content = '' ) {

		// Link format (after version 8.16)
		if (
			! empty( $atts['link'] )
			AND strpos( rawurldecode( $atts['link'] ), '{"' ) !== 0 // this check ignores new link format
		) {
			$atts['link'] = us_fallback_link_format( $atts['link'] );
		}

		// Custom Link (after version 8.16)
		if (
			! empty( $atts['custom_link'] )
			AND strpos( rawurldecode( $atts['custom_link'] ), '{"' ) !== 0 // this check ignores new link format
		) {
			$link_array['url'] = $atts['custom_link'];

			if ( ! isset( $atts['custom_link_new_tab'] ) ) {
				$link_array['target'] = '_blank';
				$link_array['rel'] = 'nofollow';
			} else {
				$atts['custom_link_new_tab'] = '';
			}

			$atts['custom_link'] = us_fallback_link_format( $link_array );
		}

		return $atts;
	}
}

// Interactive Banner
if ( ! function_exists( 'us_fallback_atts_us_ibanner' ) ) {
	add_filter( 'vc_edit_form_fields_attributes_us_ibanner', 'us_fallback_atts_us_ibanner', 710 );
	add_filter( 'us_edit_atts_fallback_us_ibanner', 'us_fallback_atts_us_ibanner', 710 );
	add_filter( 'us_fallback_atts_us_ibanner', 'us_fallback_atts_us_ibanner', 710 );
	function us_fallback_atts_us_ibanner( $atts ) {

		// Link format (after version 8.16)
		if (
			! empty( $atts['link'] )
			AND strpos( rawurldecode( $atts['link'] ), '{"' ) !== 0 // this check ignores new link format
		) {
			$atts['link'] = us_fallback_link_format( $atts['link'] );
		}

		return $atts;
	}
}

// Interactive Banner design settings
// Fallback "css" attribute separately, because it can work in the element editing only
if ( ! function_exists( 'us_fallback_edit_atts_us_ibanner' ) ) {
	add_filter( 'vc_edit_form_fields_attributes_us_ibanner', 'us_fallback_edit_atts_us_ibanner', 710 );
	add_filter( 'us_edit_atts_fallback_us_ibanner', 'us_fallback_edit_atts_us_ibanner', 710 );
	function us_fallback_edit_atts_us_ibanner( $atts ) {

		// Alignment (after version 7.1)
		if ( ! empty( $atts['align'] ) AND ( $atts['align'] != 'left' OR is_rtl() ) ) {
			if ( ! empty( $atts['css'] ) ) {
				if ( is_string( $atts['css'] ) ) {
					$css_arr = json_decode( rawurldecode( $atts['css'] ), TRUE );
				} else {
					$css_arr = $atts['css'];
				}
			} else {
				$css_arr = array();
			}

			if ( ! is_array( $css_arr ) ) {
				$css_arr = array();
			}
			if ( empty( $css_arr['default']['text-align'] ) ) {
				if ( ! isset( $css_arr['default'] ) ) {
					$css_arr['default'] = array();
				}
				$css_arr['default']['text-align'] = $atts['align'];
				$atts['css'] = rawurlencode( json_encode( $css_arr ) );
			}
			$atts['align'] = '';
		}

		return $atts;
	}
}

// Search
if ( ! function_exists( 'us_fallback_atts_us_search' ) ) {
	add_filter( 'vc_edit_form_fields_attributes_us_search', 'us_fallback_atts_us_search', 710, 1 );
	add_filter( 'us_edit_atts_fallback_us_search', 'us_fallback_atts_us_search', 710, 2 );
	add_filter( 'us_fallback_atts_us_search', 'us_fallback_atts_us_search', 710, 2 );
	function us_fallback_atts_us_search( $atts, $content = '' ) {

		// Search Shop Products only
		if ( ! empty( $atts['product_search'] ) ) {
			$atts['search_post_type'] = 'product';
			$atts['product_search'] = '';
		}

		return $atts;
	}
}

// Simple Menu design settings
// Fallback "css" attribute separately, because it can work in the element editing only
if ( ! function_exists( 'us_fallback_edit_atts_us_additional_menu' ) ) {
	add_filter( 'vc_edit_form_fields_attributes_us_additional_menu', 'us_fallback_edit_atts_us_additional_menu', 710 );
	add_filter( 'us_edit_atts_fallback_us_additional_menu', 'us_fallback_edit_atts_us_additional_menu', 710 );
	function us_fallback_edit_atts_us_additional_menu( $atts ) {

		// Alignment (after version 7.1)
		if ( ! empty( $atts['align'] ) AND ( $atts['align'] != 'left' OR is_rtl() ) ) {
			if ( ! empty( $atts['css'] ) ) {
				if ( is_string( $atts['css'] ) ) {
					$css_arr = json_decode( rawurldecode( $atts['css'] ), TRUE );
				} else {
					$css_arr = $atts['css'];
				}
			} else {
				$css_arr = array();
			}

			if ( ! is_array( $css_arr ) ) {
				$css_arr = array();
			}
			if ( empty( $css_arr['default']['text-align'] ) ) {
				if ( ! isset( $css_arr['default'] ) ) {
					$css_arr['default'] = array();
				}
				$css_arr['default']['text-align'] = $atts['align'];
				$atts['css'] = rawurlencode( json_encode( $css_arr ) );
			}
			$atts['align'] = '';
		}

		return $atts;
	}
}

// Social Links
if ( ! function_exists( 'us_fallback_atts_us_socials' ) ) {
	add_filter( 'vc_edit_form_fields_attributes_us_socials', 'us_fallback_atts_us_socials', 710, 1 );
	add_filter( 'us_edit_atts_fallback_us_socials', 'us_fallback_atts_us_socials', 710, 2 );
	add_filter( 'us_fallback_atts_us_socials', 'us_fallback_atts_us_socials', 710, 2 );
	function us_fallback_atts_us_socials( $atts, $content = '' ) {

		// Color
		if (
			! empty( $atts['color'] )
			AND $atts['color'] != 'brand'
		) {
			$atts['icons_color'] = $atts['color'];
			$atts['color'] = '';
		}

		// Links
		if ( ! empty( $atts['items'] ) ) {
			$_group_value_type = is_array( $atts['items'] ) ? 'array' : 'string';
			$items = is_array( $atts['items'] ) ? $atts['items'] : json_decode( rawurldecode( $atts['items'] ), TRUE );
			foreach ( $items as $key => $item ) {

				// Link format (after version 8.17)
				if (
					! empty( $item['url'] )
					AND strpos( rawurldecode( $item['url'] ), '{"' ) !== 0 // this check ignores new link format
				) {
					$link_atts = array(
						'url' => $item['url'],
						'target' => '_blank',
					);

					// Move separate "nofollow" value to the link value
					if ( ! isset( $atts['nofollow'] ) OR $atts['nofollow'] != '0' ) {
						$link_atts['rel'] = 'nofollow';
					} else {
						$link_atts['rel'] = '';
					}

					// Reset the old "fallback" param
					if ( isset( $atts['nofollow'] ) ) {
						$atts['nofollow'] = '';
					}

					// Email link
					if ( $item['type'] == 'email' AND is_email( $link_atts['url'] ) ) {
						$link_atts['url'] = 'mailto:' . $link_atts['url'];
						unset( $link_atts['target'] );
						$link_atts['rel'] = '';
					}

					// Skype link
					if ( $item['type'] == 'skype' AND strpos( $link_atts['url'], ':' ) === FALSE ) {
						$link_atts['url'] = 'skype:' . $link_atts['url'];
						unset( $link_atts['target'] );
						$link_atts['rel'] = '';
					}

					$items[ $key ]['url'] = us_fallback_link_format( $link_atts );
				}
			}
			$atts['items'] = ( $_group_value_type == 'array' ) ? $items : rawurlencode( json_encode( $items ) );
			unset( $items );
		}

		return $atts;
	}
}

// Social Links design settings
// Fallback "css" attribute separately, because it can work in the element editing only
if ( ! function_exists( 'us_fallback_edit_atts_us_socials' ) ) {
	add_filter( 'vc_edit_form_fields_attributes_us_socials', 'us_fallback_edit_atts_us_socials', 710 );
	add_filter( 'us_edit_atts_fallback_us_socials', 'us_fallback_edit_atts_us_socials', 710 );
	function us_fallback_edit_atts_us_socials( $atts ) {

		// Alignment (after version 7.1)
		if ( ! empty( $atts['align'] ) AND ( $atts['align'] != 'left' OR is_rtl() ) ) {
			if ( ! empty( $atts['css'] ) ) {
				if ( is_string( $atts['css'] ) ) {
					$css_arr = json_decode( rawurldecode( $atts['css'] ), TRUE );
				} else {
					$css_arr = $atts['css'];
				}
			} else {
				$css_arr = array();
			}

			if ( ! is_array( $css_arr ) ) {
				$css_arr = array();
			}

			if ( empty( $css_arr['default']['text-align'] ) ) {
				if ( ! isset( $css_arr['default'] ) ) {
					$css_arr['default'] = array();
				}
				$css_arr['default']['text-align'] = $atts['align'];
				$atts['css'] = rawurlencode( json_encode( $css_arr ) );
			}
			$atts['align'] = '';
		}

		return $atts;
	}
}

// VC Raw HTML -> US Custom HTML (change the shortcode name only)
if ( ! function_exists( 'usb_fallback_name_vc_raw_html' ) ) {
	add_filter( 'usb_fallback_name_vc_raw_html', 'usb_fallback_name_vc_raw_html', 710, 0 );
	function usb_fallback_name_vc_raw_html() {
		return 'us_html';
	}
}

// Video Player
if ( ! function_exists( 'us_fallback_atts_vc_video' ) ) {
	add_filter( 'vc_edit_form_fields_attributes_vc_video', 'us_fallback_atts_vc_video', 710, 1 );
	add_filter( 'us_edit_atts_fallback_vc_video', 'us_fallback_atts_vc_video', 710, 2 );
	add_filter( 'us_fallback_atts_vc_video', 'us_fallback_atts_vc_video', 710, 2 );
	function us_fallback_atts_vc_video( $atts, $content = '' ) {
		// Controls (after version 8.14)
		if ( ! empty( $atts['hide_controls'] ) ) {
			$atts['controls'] = '0';
			$atts['hide_controls'] = '';
		}

		// Change source to custom fields (after version 8.20)
		if ( ! empty( $atts['source'] ) AND $atts['source'] !== 'custom' ) {
			$atts['link'] = '{{' . $atts['source'] . '}}';
			unset( $atts['source'] );
		}

		return $atts;
	}
}

// US Grid
if ( ! function_exists( 'us_fallback_atts_us_grid' ) ) {
	add_filter( 'vc_edit_form_fields_attributes_us_grid', 'us_fallback_atts_us_grid', 710, 1 );
	add_filter( 'us_edit_atts_fallback_us_grid', 'us_fallback_atts_us_grid', 710, 2 );
	add_filter( 'us_fallback_atts_us_grid', 'us_fallback_atts_us_grid', 710, 2 );
	/**
	 * @param array $atts
	 * @return array
	 */
	function us_fallback_atts_us_grid( $atts, $content = '' ) {

		// Order by (after version 7.11)
		if ( isset( $atts['orderby'] ) AND $atts['orderby'] == 'alpha' ) {
			$atts['orderby'] = 'title';
		}

		// Replace taxonomy identifiers with slug
		foreach ( (array) $atts as $key => $values ) {
			if ( strpos( $key, 'taxonomy_' ) === 0 AND ! empty( $values ) ) {

				$values = explode( ',', $values );
				$taxonomy = substr( $key, strlen( 'taxonomy_' ) );

				foreach ( $values as &$value ) {
					if ( is_numeric( $value ) AND $term = get_term( (int) $value, $taxonomy ) ) {
						$value = $term->slug;
					}
				}
				unset( $value );

				$atts[ $key ] = implode( ',', $values );
			}
		}

		// Overriding Link (after version 8.16)
		if ( ! empty( $atts['overriding_link'] ) ) {
			if ( $atts['overriding_link'] == 'post' ) {
				$link_array = array( 'type' => 'post' );

			} elseif ( $atts['overriding_link'] == 'popup_post' ) {
				$link_array = array( 'type' => 'popup_post' );

			} elseif ( $atts['overriding_link'] == 'popup_post_image' ) {
				$link_array = array( 'type' => 'popup_image' );
			}

			if ( ! empty( $link_array ) ) {
				$atts['overriding_link'] = us_fallback_link_format( $link_array );
			}
		}

		return $atts;
	}
}

if ( ! function_exists( 'us_fallback_metabox_value' ) ) {
	add_filter( 'us_fallback_metabox_value', 'us_fallback_metabox_value', 1, 3 );
	/**
	 * Applies changes to meta value for compatibility with old theme versions
	 *
	 * @param mixed $meta_value The meta value
	 * @param string $meta_key The meta key
	 * @param array $field The field options
	 * @return mixed Returns compatible meta value for different versions
	 */
	function us_fallback_metabox_value( $meta_value, $meta_key = '', $field = array() ) {

		if ( ! is_array( $field ) OR ! isset( $field['type'] ) ) {
			return $meta_value;
		}

		// "Do not display" or "Show content as is" value shouldn't be empty string (after version 8.14)
		if (
			$field['type'] == 'select'
			AND $meta_value == ''
			AND preg_match( '/^us_([a-z\_]+)_id$/', $meta_key )
		) {
			return '0';
		}

		// Always return integer value for the "switch" type
		if ( $field['type'] == 'switch' ) {
			return (int) $meta_value;
		}

		return $meta_value;
	}
}

// US Map
if ( ! function_exists( 'us_fallback_atts_us_gmaps' ) ) {
	add_filter( 'vc_edit_form_fields_attributes_us_gmaps', 'us_fallback_atts_us_gmaps', 710, 1 );
	add_filter( 'us_edit_atts_fallback_us_gmaps', 'us_fallback_atts_us_gmaps', 710, 2 );
	add_filter( 'us_fallback_atts_us_gmaps', 'us_fallback_atts_us_gmaps', 710, 2 );
	/**
	 * @param array $atts
	 * @return array
	 */
	function us_fallback_atts_us_gmaps( $atts, $content = '' ) {
		if ( ! empty( $atts['source'] ) AND $atts['source'] !== 'custom' ) {
			$atts['marker_address'] = '{{' . $atts['source'] . '}}';
			unset( $atts['source'] );
		}

		return $atts;
	}
}

// US Gallery
if ( ! function_exists( 'us_fallback_atts_us_gallery' ) ) {
	add_filter( 'vc_edit_form_fields_attributes_us_gallery', 'us_fallback_atts_us_gallery', 710, 1 );
	add_filter( 'us_edit_atts_fallback_us_gallery', 'us_fallback_atts_us_gallery', 710, 2 );
	add_filter( 'us_fallback_atts_us_gallery', 'us_fallback_atts_us_gallery', 710, 2 );
	/**
	 * @param array $atts
	 * @return array
	 */
	function us_fallback_atts_us_gallery( $atts, $content = '' ) {
		if ( ! empty( $atts['source'] ) ) {
			if ( strpos( $atts['source'], 'cf|' ) === 0 ) {
				$atts['ids'] = '{{' . str_replace( 'cf|', '', $atts['source'] ) . '}}';
			}
			unset( $atts['source'] );
		}

		if ( ! empty( $atts['show_all'] ) ) {
			$atts['quantity_type'] = 'all';
			unset( $atts['show_all'] );
		}

		return $atts;
	}
}

// Display Logic for ALL theme elements
if ( ! function_exists( 'us_fallback_atts_display_conditions' ) ) {

	add_action( 'init', function() {
		foreach( us_config( 'shortcodes.theme_elements' ) as $elm_name ) {
			$shortcode = us_get_shortcode_full_name( $elm_name );

			add_filter( 'vc_edit_form_fields_attributes_' . $shortcode, 'us_fallback_atts_display_conditions', 720 );
			add_filter( 'us_edit_atts_fallback_' . $shortcode, 'us_fallback_atts_display_conditions', 720 );
		}
	}, 401 );

	function us_fallback_atts_display_conditions( $atts ) {

		// Conditions
		if (
			! empty( $atts['conditions'] )
			AND ! empty( $atts['conditions_operator'] )
			AND in_array( $atts['conditions_operator'], array( 'and', 'or' ) )
		) {
			$_group_value_type = is_array( $atts['conditions'] ) ? 'array' : 'string';
			$conditions = is_array( $atts['conditions'] ) ? $atts['conditions'] : json_decode( rawurldecode( $atts['conditions'] ), TRUE );

			foreach ( $conditions as $i => &$condition ) {
				if ( ! $condition_param = us_arr_path( $condition, 'param' ) ) {
					continue;
				}

				// Date / Time (after version 8.28.1)
				if ( $condition_param == 'time' ) {
					switch ( us_arr_path( $condition, 'time_operator', 'since' ) ) {
						case 'dm':
							$condition['time_year'] = 'any';
							$condition['time_weekday'] = 'any';
							$condition['time_hour'] = 'any';
							$condition['time_minute'] = 'any';
							$condition['time_operator'] = '=';
							break;

						case 'd':
							$condition['time_year'] = 'any';
							$condition['time_month'] = 'any';
							$condition['time_weekday'] = 'any';
							$condition['time_hour'] = 'any';
							$condition['time_minute'] = 'any';
							$condition['time_operator'] = '=';
							break;

						case 'm':
							$condition['time_year'] = 'any';
							$condition['time_day'] = 'any';
							$condition['time_weekday'] = 'any';
							$condition['time_hour'] = 'any';
							$condition['time_minute'] = 'any';
							$condition['time_operator'] = '=';
							break;

						case 'w':
							$condition['time_year'] = 'any';
							$condition['time_month'] = 'any';
							$condition['time_day'] = 'any';
							$condition['time_hour'] = 'any';
							$condition['time_minute'] = 'any';
							$condition['time_operator'] = '=';
							break;
					}
				}
			}
			$atts['conditions'] = ( $_group_value_type == 'array' ) ? $conditions : rawurlencode( json_encode( $conditions ) );
			unset( $conditions );
		}

		return $atts;
	}
}
